<?php
require __DIR__ . '/_init.php';
require_login();
require_perm('dashboard','view');

$pdo = db();

/**
 * Filtros:
 *  - period=mensal | quinzenal | custom
 *  - mensal: month=YYYY-MM
 *  - quinzenal: month=YYYY-MM & half=1|2
 *  - custom: start=YYYY-MM-DD & end=YYYY-MM-DD
 */
$period = $_GET['period'] ?? 'mensal';

$today = new DateTime('now');
$monthParam = $_GET['month'] ?? $today->format('Y-m');
$half = (int)($_GET['half'] ?? 1);
$startParam = $_GET['start'] ?? $today->format('Y-m-01');
$endParam = $_GET['end'] ?? $today->format('Y-m-t');

function month_bounds(string $ym): array {
  if (!preg_match('/^\d{4}-\d{2}$/', $ym)) {
    $d = new DateTime('first day of this month');
    return [$d->format('Y-m-01'), $d->format('Y-m-t')];
  }
  $d = DateTime::createFromFormat('Y-m-d', $ym.'-01');
  return [$d->format('Y-m-01'), $d->format('Y-m-t')];
}

function half_bounds(string $ym, int $half): array {
  [$mstart, $mend] = month_bounds($ym);
  $dStart = new DateTime($mstart);
  $dEnd = new DateTime($mend);
  $half = ($half === 2) ? 2 : 1;
  if ($half === 1) return [$dStart->format('Y-m-01'), $dStart->format('Y-m-15')];
  $y = (int)$dStart->format('Y');
  $m = (int)$dStart->format('m');
  $last = (int)$dEnd->format('d');
  return [sprintf('%04d-%02d-16', $y, $m), sprintf('%04d-%02d-%02d', $y, $m, $last)];
}

function valid_date(string $s): bool {
  return (bool)preg_match('/^\d{4}-\d{2}-\d{2}$/', $s);
}

function fmt_money(float $v): string {
  return 'R$ '.number_format($v, 2, ',', '.');
}

// resolve período
$startDate = $today->format('Y-m-01');
$endDate   = $today->format('Y-m-t');

if ($period === 'custom') {
  $startDate = valid_date($startParam) ? $startParam : $today->format('Y-m-01');
  $endDate   = valid_date($endParam) ? $endParam : $today->format('Y-m-t');
  if (strtotime($endDate) < strtotime($startDate)) [$startDate, $endDate] = [$endDate, $startDate];
} elseif ($period === 'quinzenal') {
  $half = ($half === 2) ? 2 : 1;
  [$startDate, $endDate] = half_bounds($monthParam, $half);
} else {
  $period = 'mensal';
  [$startDate, $endDate] = month_bounds($monthParam);
}

$startDT = $startDate.' 00:00:00';
$endDT   = $endDate.' 23:59:59';

// Totais realizados
$real = $pdo->prepare("
  SELECT
    COALESCE(SUM(CASE WHEN entry_type='CREDITO' THEN amount ELSE 0 END),0) AS entradas,
    COALESCE(SUM(CASE WHEN entry_type='DEBITO' THEN amount ELSE 0 END),0) AS saidas
  FROM bank_ledger
  WHERE occurred_at BETWEEN ? AND ?
");
$real->execute([$startDT,$endDT]);
$realRow = $real->fetch() ?: ['entradas'=>0,'saidas'=>0];

$entradas = (float)$realRow['entradas'];
$saidas   = (float)$realRow['saidas'];
$liquido  = $entradas - $saidas;

// Em aberto no período (por vencimento)
$open = $pdo->prepare("
  SELECT
    COALESCE(SUM(CASE WHEN d.doc_type='PAGAR' THEN (i.amount - i.paid_amount) ELSE 0 END),0) AS pagar_aberto,
    COALESCE(SUM(CASE WHEN d.doc_type='RECEBER' THEN (i.amount - i.paid_amount) ELSE 0 END),0) AS receber_aberto
  FROM fin_installments i
  JOIN fin_documents d ON d.id=i.document_id
  WHERE i.status IN ('ABERTO','PARCIAL')
    AND i.due_date BETWEEN ? AND ?
");
$open->execute([$startDate, $endDate]);
$openRow = $open->fetch() ?: ['pagar_aberto'=>0,'receber_aberto'=>0];

$pagarAberto = (float)$openRow['pagar_aberto'];
$receberAberto = (float)$openRow['receber_aberto'];

// Série diária (sparklines) - agregado por dia
$daily = $pdo->prepare("
  SELECT DATE(occurred_at) AS d,
    COALESCE(SUM(CASE WHEN entry_type='CREDITO' THEN amount ELSE 0 END),0) AS credit,
    COALESCE(SUM(CASE WHEN entry_type='DEBITO' THEN amount ELSE 0 END),0) AS debit
  FROM bank_ledger
  WHERE occurred_at BETWEEN ? AND ?
  GROUP BY DATE(occurred_at)
  ORDER BY d
");
$daily->execute([$startDT,$endDT]);
$dailyRows = $daily->fetchAll();

// Completa dias faltantes com 0
$startObj = new DateTime($startDate);
$endObj = new DateTime($endDate);
$map = [];
foreach ($dailyRows as $r) $map[$r['d']] = ['c'=>(float)$r['credit'], 'd'=>(float)$r['debit']];
$seriesC = [];
$seriesD = [];
$cursor = clone $startObj;
while ($cursor <= $endObj) {
  $key = $cursor->format('Y-m-d');
  $seriesC[] = $map[$key]['c'] ?? 0.0;
  $seriesD[] = $map[$key]['d'] ?? 0.0;
  $cursor->modify('+1 day');
}

function sparkline_points(array $values, int $w=160, int $h=44, int $pad=4): string {
  $n = count($values);
  if ($n <= 1) return "";
  $min = min($values);
  $max = max($values);
  if (abs($max - $min) < 0.00001) { $max = $min + 1.0; } // evita divisão por zero
  $dx = ($w - 2*$pad) / ($n - 1);
  $pts = [];
  for ($i=0; $i<$n; $i++) {
    $x = $pad + $dx*$i;
    $t = ($values[$i] - $min) / ($max - $min);
    $y = ($h - $pad) - $t * ($h - 2*$pad);
    $pts[] = round($x,2).",".round($y,2);
  }
  return implode(" ", $pts);
}

$ptsC = sparkline_points($seriesC);
$ptsD = sparkline_points($seriesD);

$banks = $pdo->query("SELECT * FROM banks ORDER BY bank_name")->fetchAll();

render_header('Dashboard');
?>

<div class="dash-bg"></div>

<div class="dash-hero mb-3">
  <div class="dash-title">
    <div class="dash-kicker">PAINEL FINANCEIRO</div>
    <h2 class="m-0">Dashboard</h2>
    <div class="dash-sub">
      Período: <b><?=h(date('d/m/Y', strtotime($startDate)))?></b> a <b><?=h(date('d/m/Y', strtotime($endDate)))?></b>
    </div>
  </div>

  <div class="dash-actions">
    <?php
      $thisMonth = (new DateTime('first day of this month'))->format('Y-m');
      $prevMonth = (new DateTime('first day of last month'))->format('Y-m');
      $curMonth = $today->format('Y-m');

      // Define qual atalho está ativo (0..3) ou -1
      $segIndex = -1;
      if ($period==='mensal' && $monthParam===$thisMonth) $segIndex = 0;
      else if ($period==='mensal' && $monthParam===$prevMonth) $segIndex = 1;
      else if ($period==='quinzenal' && $monthParam===$curMonth && $half===1) $segIndex = 2;
      else if ($period==='quinzenal' && $monthParam===$curMonth && $half===2) $segIndex = 3;

      $href0 = "?period=mensal&month=".h($thisMonth);
      $href1 = "?period=mensal&month=".h($prevMonth);
      $href2 = "?period=quinzenal&month=".h($curMonth)."&half=1";
      $href3 = "?period=quinzenal&month=".h($curMonth)."&half=2";
      $href4 = "assistant_ai.php?start=".h($startDate)."&end=".h($endDate); // análise IA do mesmo período
    ?>
    <div class="dash-seg" style="--seg-count:5; --seg-index: <?=$segIndex?>;">
      <div class="dash-seg-title"><i class="bi bi-command"></i> Atalhos rápidos <span class="dash-seg-hint">Alt+1..4</span></div>
      <div class="dash-seg-bar" role="tablist" aria-label="Atalhos rápidos">
        <span class="seg-indicator" aria-hidden="true"></span>

        <a class="seg-btn ripple-target <?=$segIndex===0?'active':''?>" href="<?=$href0?>" data-hotkey="1" data-href="<?=$href0?>" role="tab" title="Este mês (Alt+1)" aria-selected="<?=$segIndex===0?'true':'false'?>">
          <span class="seg-ico"><i class="bi bi-calendar2-week"></i></span>
          <span class="seg-text">Este mês</span>
        </a>

        <a class="seg-btn ripple-target <?=$segIndex===1?'active':''?>" href="<?=$href1?>" data-hotkey="2" data-href="<?=$href1?>" role="tab" title="Mês anterior (Alt+2)" aria-selected="<?=$segIndex===1?'true':'false'?>">
          <span class="seg-ico"><i class="bi bi-arrow-left-circle"></i></span>
          <span class="seg-text">Mês anterior</span>
        </a>

        <a class="seg-btn ripple-target <?=$segIndex===2?'active':''?>" href="<?=$href2?>" data-hotkey="3" data-href="<?=$href2?>" role="tab" title="1ª quinzena (Alt+3)" aria-selected="<?=$segIndex===2?'true':'false'?>">
          <span class="seg-ico"><i class="bi bi-brightness-high"></i></span>
          <span class="seg-text">1ª quinzena</span>
        </a>

        <a class="seg-btn ripple-target <?=$segIndex===3?'active':''?>" href="<?=$href3?>" data-hotkey="4" data-href="<?=$href3?>" role="tab" title="2ª quinzena (Alt+4)" aria-selected="<?=$segIndex===3?'true':'false'?>">
          <span class="seg-ico"><i class="bi bi-moon-stars"></i></span>
          <span class="seg-text">2ª quinzena</span>
        </a>

        <a class="seg-btn ripple-target" href="<?=$href4?>" data-hotkey="5" data-href="<?=$href4?>" role="tab" aria-selected="false" title="Análise IA (Alt+5)">
          <span class="seg-ico"><i class="bi bi-stars"></i></span>
          <span class="seg-text">Análise IA</span>
        </a>
      </div>
    </div>
  </div>
</div>

<form class="card dash-filters shadow-sm p-3 mb-3" method="get">
  <div class="row g-3 align-items-end">
    <div class="col-md-3">
      <label class="form-label">Tipo de filtro</label>
      <select class="form-select" name="period" id="periodSelect">
        <option value="mensal" <?=$period==='mensal'?'selected':''?>>Mensal</option>
        <option value="quinzenal" <?=$period==='quinzenal'?'selected':''?>>Quinzenal</option>
        <option value="custom" <?=$period==='custom'?'selected':''?>>Por data</option>
      </select>
    </div>

    <div class="col-md-3 period-mensal period-quinzenal">
      <label class="form-label">Mês</label>
      <input class="form-control" type="month" name="month" value="<?=h($monthParam)?>">
    </div>

    <div class="col-md-3 period-quinzenal">
      <label class="form-label">Quinzena</label>
      <select class="form-select" name="half">
        <option value="1" <?=$half===1?'selected':''?>>1ª (1–15)</option>
        <option value="2" <?=$half===2?'selected':''?>>2ª (16–fim)</option>
      </select>
    </div>

    <div class="col-md-3 period-custom">
      <label class="form-label">Início</label>
      <input class="form-control" type="date" name="start" value="<?=h($startDate)?>">
    </div>

    <div class="col-md-3 period-custom">
      <label class="form-label">Fim</label>
      <input class="form-control" type="date" name="end" value="<?=h($endDate)?>">
    </div>

    <div class="col-md-3 d-grid">
      <button class="btn btn-outline-dark ripple-target">Aplicar</button>
    </div>
  </div>
</form>

<script>
  (function(){
    function apply(){
      const v = document.getElementById('periodSelect').value;
      document.querySelectorAll('.period-mensal').forEach(el=> el.style.display = (v==='mensal' || v==='quinzenal') ? '' : 'none');
      document.querySelectorAll('.period-quinzenal').forEach(el=> el.style.display = (v==='quinzenal') ? '' : 'none');
      document.querySelectorAll('.period-custom').forEach(el=> el.style.display = (v==='custom') ? '' : 'none');
    }
    document.getElementById('periodSelect').addEventListener('change', apply);
    apply();
  })();
</script>

<div class="row g-3 mb-3">
  <div class="col-md-3">
    <div class="kpi-card kpi-in" data-tilt>
      <div class="kpi-top">
        <div class="kpi-label">Entradas</div>
        <div class="kpi-icon"><i class="bi bi-arrow-down-left-circle"></i></div>
      </div>
      <div class="kpi-value" data-countup="<?=h((string)$entradas)?>">0</div>
      <div class="kpi-sub">Realizado no período</div>
      <svg class="kpi-spark" viewBox="0 0 160 44" preserveAspectRatio="none" aria-hidden="true">
        <polyline class="spark-line spark-in" points="<?=h($ptsC)?>" />
      </svg>
      <div class="kpi-glow"></div>
    </div>
  </div>

  <div class="col-md-3">
    <div class="kpi-card kpi-out" data-tilt>
      <div class="kpi-top">
        <div class="kpi-label">Saídas</div>
        <div class="kpi-icon"><i class="bi bi-arrow-up-right-circle"></i></div>
      </div>
      <div class="kpi-value" data-countup="<?=h((string)$saidas)?>">0</div>
      <div class="kpi-sub">Realizado no período</div>
      <svg class="kpi-spark" viewBox="0 0 160 44" preserveAspectRatio="none" aria-hidden="true">
        <polyline class="spark-line spark-out" points="<?=h($ptsD)?>" />
      </svg>
      <div class="kpi-glow"></div>
    </div>
  </div>

  <div class="col-md-3">
    <div class="kpi-card kpi-net" data-tilt>
      <div class="kpi-top">
        <div class="kpi-label">Saldo líquido</div>
        <div class="kpi-icon"><i class="bi bi-lightning-charge"></i></div>
      </div>
      <div class="kpi-value" data-countup="<?=h((string)$liquido)?>">0</div>
      <div class="kpi-sub">Entradas − Saídas</div>
      <div class="kpi-micro">
        <span class="chip">Entradas: <b><?=h(fmt_money($entradas))?></b></span>
        <span class="chip">Saídas: <b><?=h(fmt_money($saidas))?></b></span>
      </div>
      <div class="kpi-glow"></div>
    </div>
  </div>

  <div class="col-md-3">
    <div class="kpi-card kpi-open" data-tilt>
      <div class="kpi-top">
        <div class="kpi-label">Em aberto</div>
        <div class="kpi-icon"><i class="bi bi-hourglass-split"></i></div>
      </div>
      <div class="kpi-stack">
        <div class="kpi-row">
          <span class="kpi-row-label">Pagar</span>
          <span class="kpi-row-val" data-countup="<?=h((string)$pagarAberto)?>">0</span>
        </div>
        <div class="kpi-row">
          <span class="kpi-row-label">Receber</span>
          <span class="kpi-row-val" data-countup="<?=h((string)$receberAberto)?>">0</span>
        </div>
      </div>
      <div class="kpi-sub">Vencimentos dentro do período</div>
      <div class="kpi-glow"></div>
    </div>
  </div>
</div>

<div class="dash-section d-flex justify-content-between align-items-center mb-2">
  <h5 class="mb-0">Saldos dos Bancos</h5>
  <div class="text-muted small">Saldo atual por conta</div>
</div>

<div class="row g-3">
  <?php foreach ($banks as $b): ?>
    <div class="col-md-4">
      <div class="bank-card" data-tilt>
        <div class="bank-head">
          <div>
            <div class="bank-name"><?=h($b['bank_name'])?> <span class="bank-num">(<?=h($b['bank_number'])?>)</span></div>
            <div class="bank-sub">Ag <?=h($b['agency'])?> • <?=h($b['account_type'])?></div>
          </div>
          <div class="bank-badge"><i class="bi bi-bank2"></i></div>
        </div>
        <div class="bank-balance" data-countup="<?=h((string)$b['current_balance'])?>">0</div>
        <div class="bank-foot">Atualizado: <?=h($b['updated_at'] ? date('d/m/Y H:i', strtotime($b['updated_at'])) : date('d/m/Y H:i', strtotime($b['created_at'])))?></div>
        <div class="bank-glow"></div>
      </div>
    </div>
  <?php endforeach; ?>
  <?php if (!$banks): ?>
    <div class="col-12"><div class="alert alert-warning">Nenhum banco cadastrado ainda.</div></div>
  <?php endif; ?>
</div>

<?php render_footer(); ?>
