<?php
// app/helpers.php
function h(string $s): string { return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }

/**
 * Base URL resiliente.
 * - Se app.base_url estiver definido, usa ele
 * - Caso contrário, detecta automaticamente a partir de /public (dirname de SCRIPT_NAME)
 */
function perm_can(string $key, string $action='view'): bool {
  $role = $_SESSION['user_role'] ?? '';
  if ($role === 'admin') return true;
  $perms = $_SESSION['perms'] ?? [];
  $p = $perms[$key] ?? null;
  if (!$p) return false;
  return (bool)($p[$action] ?? false);
}

function require_perm(string $key, string $action='view'): void {
  if (!perm_can($key, $action)) {
    http_response_code(403);
    die('Sem permissão.');
  }
}

function base_url_runtime(): string {
  $cfg = require __DIR__ . '/config.php';
  $forced = rtrim((string)($cfg['app']['base_url'] ?? ''), '/');
  if ($forced !== '') return $forced;
  $script = (string)($_SERVER['SCRIPT_NAME'] ?? '');
  $dir = rtrim(str_replace('\\', '/', dirname($script)), '/');
  if ($dir === '.' || $dir === '/') return '';
  return $dir;
}


function redirect(string $path) {
  $base = base_url_runtime();
  header("Location: {$base}{$path}");
  exit;
}

function csrf_token(): string {
  if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(32));
  return $_SESSION['csrf'];
}

function csrf_check(): void {
  $token = $_POST['csrf'] ?? '';
  if (!$token || empty($_SESSION['csrf']) || !hash_equals($_SESSION['csrf'], $token)) {
    http_response_code(403);
    die('CSRF inválido.');
  }
}

function money_to_decimal(string $s): string {
  $s = trim($s);
  $s = str_replace(['R$', ' '], '', $s);
  if (str_contains($s, ',')) {
    $s = str_replace('.', '', $s);
    $s = str_replace(',', '.', $s);
  }
  if (!is_numeric($s)) return '0.00';
  return number_format((float)$s, 2, '.', '');
}

function render_header(string $title): void {
  $base = base_url_runtime();

  $appName = function_exists('setting_get') ? (string)(setting_get('app.name', 'Financeiro') ?: 'Financeiro') : 'Financeiro';
  $logoPath = function_exists('setting_get') ? (string)(setting_get('app.logo_path', '') ?: '') : '';

  $page = basename($_SERVER['PHP_SELF'] ?? '');
  $qs = $_SERVER['QUERY_STRING'] ?? '';
  $full = $page . ($qs ? ('?'.$qs) : '');

  $isActive = function(string $needle) use ($page, $full): bool {
    return str_contains($page, $needle) || str_contains($full, $needle);
  };

  echo '<!doctype html><html lang="pt-br" data-theme="dark"><head>';
  echo '<meta charset="utf-8"><meta name="viewport" content="width=device-width, initial-scale=1">';
  echo '<title>'.h($title).'</title>';
  echo '<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">';
  echo '<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">';
  echo '<link href="'.$base.'/assets/app.css" rel="stylesheet">';
  echo '</head><body class="app-shell no-js">';

  echo '<div class="app-overlay" id="appOverlay"></div>';

  echo '<aside class="app-sidebar" id="appSidebar">';
  echo '  <div class="sidebar-brand">';
  echo '    <a class="brand-link ripple-target" href="'.$base.'/index.php">';
  if ($logoPath) echo '      <img class="brand-logo" src="'.h($base.$logoPath).'" alt="logo">';
  echo '      <span class="brand-name">'.h($appName).'</span>';
  echo '    </a>';
  echo '    <div class="brand-actions">';
  echo '      <button class="btn btn-sm btn-ghost icon-btn ripple-target d-none d-lg-inline" id="btnCollapse" type="button" title="Recolher">';
  echo '        <i class="bi bi-layout-sidebar-inset"></i>';
  echo '      </button>';
  echo '    </div>';
  echo '  </div>';

  echo '  <div class="sidebar-search px-3 pt-2 pb-2">';
  echo '    <div class="input-group input-group-sm">';
  echo '      <span class="input-group-text bg-transparent text-white-50 border-0"><i class="bi bi-search"></i></span>';
  echo '      <input class="form-control form-control-sm sidebar-search-input" placeholder="Buscar..." id="sidebarSearch">';
  echo '    </div>';
  echo '  </div>';

  echo '  <nav class="sidebar-nav" id="sidebarNav">';

  // ===== Financeiro =====
  echo '    <div class="nav-group open" data-group="financeiro">';
  echo '      <button class="nav-group-toggle ripple-target" type="button" data-group-btn="financeiro">';
  echo '        <span class="nav-ico"><i class="bi bi-cash-stack"></i></span>';
  echo '        <span class="nav-text">Financeiro</span>';
  echo '        <span class="nav-right"><i class="bi bi-chevron-down nav-chevron"></i></span>';
  echo '      </button>';
  echo '      <div class="nav-group-items">';

  if (perm_can('contas_pagar','view')) {
    echo '        <a class="'.(str_contains($full,'contas.php?type=PAGAR')?'nav-item-link active':'nav-item-link').' ripple-target" href="'.$base.'/contas.php?type=PAGAR">';
    echo '          <span class="nav-ico"><i class="bi bi-arrow-down-circle"></i></span>';
    echo '          <span class="nav-text">Contas a Pagar</span>';
    echo '          <span class="nav-glow"></span>';
    echo '        </a>';
  }
  if (perm_can('contas_receber','view')) {
    echo '        <a class="'.(str_contains($full,'contas.php?type=RECEBER')?'nav-item-link active':'nav-item-link').' ripple-target" href="'.$base.'/contas.php?type=RECEBER">';
    echo '          <span class="nav-ico"><i class="bi bi-arrow-up-circle"></i></span>';
    echo '          <span class="nav-text">Contas a Receber</span>';
    echo '          <span class="nav-glow"></span>';
    echo '        </a>';
  }
  echo '      </div>';
  echo '    </div>';

  // ===== Cadastros =====
  echo '    <div class="nav-group open" data-group="cadastros">';
  echo '      <button class="nav-group-toggle ripple-target" type="button" data-group-btn="cadastros">';
  echo '        <span class="nav-ico"><i class="bi bi-ui-checks-grid"></i></span>';
  echo '        <span class="nav-text">Cadastros</span>';
  echo '        <span class="nav-right"><i class="bi bi-chevron-down nav-chevron"></i></span>';
  echo '      </button>';
  echo '      <div class="nav-group-items">';

  $cad = [
    ['href'=>$base.'/banks.php', 'label'=>'Bancos', 'icon'=>'bi-bank', 'active'=>$isActive('banks.php'), 'perm'=>'banks'],
    ['href'=>$base.'/parties.php', 'label'=>'Clientes/Fornecedores', 'icon'=>'bi-people', 'active'=>$isActive('parties.php'), 'perm'=>'parties'],
    ['href'=>$base.'/parties_import.php', 'label'=>'Importar CSV', 'icon'=>'bi-upload', 'active'=>$isActive('parties_import.php'), 'perm'=>'parties_import'],
    ['href'=>$base.'/categories.php', 'label'=>'Categorias', 'icon'=>'bi-tags', 'active'=>$isActive('categories.php'), 'perm'=>'categories'],
    ['href'=>$base.'/cost_centers.php', 'label'=>'Centros de Custo', 'icon'=>'bi-diagram-3', 'active'=>$isActive('cost_centers.php'), 'perm'=>'cost_centers'],
    ['href'=>$base.'/accounts.php', 'label'=>'Plano de Contas', 'icon'=>'bi-list-check', 'active'=>$isActive('accounts.php'), 'perm'=>'accounts'],

    // Administração dentro de Cadastros
    ['href'=>$base.'/users.php', 'label'=>'Usuários', 'icon'=>'bi-person-badge', 'active'=>$isActive('users.php'), 'perm'=>'users'],
    ['href'=>$base.'/permission_profiles.php', 'label'=>'Perfis de acesso', 'icon'=>'bi-sliders', 'active'=>$isActive('permission_profiles.php'), 'perm'=>'permission_profiles'],
    ['href'=>$base.'/audit_logs.php', 'label'=>'Auditoria', 'icon'=>'bi-clipboard-data', 'active'=>$isActive('audit_logs.php'), 'perm'=>'audit_logs'],
    ['href'=>$base.'/diagnose.php', 'label'=>'Diagnóstico', 'icon'=>'bi-bug', 'active'=>$isActive('diagnose.php'), 'perm'=>'audit_logs'],
  ];

  foreach ($cad as $l) {
    if (!empty($l['perm']) && !perm_can($l['perm'], 'view')) continue;
    $cls = $l['active'] ? 'nav-item-link active' : 'nav-item-link';
    echo '        <a class="'.$cls.' ripple-target" href="'.h($l['href']).'">';
    echo '          <span class="nav-ico"><i class="bi '.h($l['icon']).'"></i></span>';
    echo '          <span class="nav-text">'.h($l['label']).'</span>';
    echo '          <span class="nav-glow"></span>';
    echo '        </a>';
  }

  echo '      </div>';
  echo '    </div>';

  // ===== Relatórios =====
  echo '    <div class="nav-group" data-group="relatorios">';
  echo '      <button class="nav-group-toggle ripple-target" type="button" data-group-btn="relatorios">';
  echo '        <span class="nav-ico"><i class="bi bi-bar-chart-line"></i></span>';
  echo '        <span class="nav-text">Relatórios</span>';
  echo '        <span class="nav-right"><i class="bi bi-chevron-down nav-chevron"></i></span>';
  echo '      </button>';
  echo '      <div class="nav-group-items">';

  $rep = [
    ['href'=>$base.'/index.php', 'label'=>'Dashboard', 'icon'=>'bi-speedometer2', 'active'=>$isActive('index.php'), 'perm'=>'dashboard'],
    ['href'=>$base.'/dre.php', 'label'=>'DRE', 'icon'=>'bi-graph-up', 'active'=>$isActive('dre.php'), 'perm'=>'dre'],
    ['href'=>$base.'/reconcile_import.php', 'label'=>'Conciliação', 'icon'=>'bi-check2-circle', 'active'=>$isActive('reconcile_'), 'perm'=>'reconcile'],
  ];

  foreach ($rep as $l) {
    if (!empty($l['perm']) && !perm_can($l['perm'], 'view')) continue;
    $cls = $l['active'] ? 'nav-item-link active' : 'nav-item-link';
    echo '        <a class="'.$cls.' ripple-target" href="'.h($l['href']).'">';
    echo '          <span class="nav-ico"><i class="bi '.h($l['icon']).'"></i></span>';
    echo '          <span class="nav-text">'.h($l['label']).'</span>';
    echo '          <span class="nav-glow"></span>';
    echo '        </a>';
  }

  echo '      </div>';
  echo '    </div>';

  // ===== IA =====
  echo '    <div class="nav-group" data-group="ia">';
  echo '      <button class="nav-group-toggle ripple-target" type="button" data-group-btn="ia">';
  echo '        <span class="nav-ico"><i class="bi bi-stars"></i></span>';
  echo '        <span class="nav-text">IA</span>';
  echo '        <span class="nav-right"><i class="bi bi-chevron-down nav-chevron"></i></span>';
  echo '      </button>';
  echo '      <div class="nav-group-items">';

  $ia = [
    ['href'=>$base.'/assistant_ai.php', 'label'=>'Assistente IA', 'icon'=>'bi-robot', 'active'=>$isActive('assistant_ai.php'), 'perm'=>'assistant_ai'],
    ['href'=>$base.'/prompt_categories.php', 'label'=>'Categorias de Prompt', 'icon'=>'bi-folder2', 'active'=>$isActive('prompt_categories.php'), 'perm'=>'prompt_categories'],
    ['href'=>$base.'/prompts.php', 'label'=>'Prompts', 'icon'=>'bi-card-text', 'active'=>$isActive('prompts.php'), 'perm'=>'prompts'],
  ];
  foreach ($ia as $l) {
    if (!empty($l['perm']) && !perm_can($l['perm'], 'view')) continue;
    $cls = $l['active'] ? 'nav-item-link active' : 'nav-item-link';
    echo '        <a class="'.$cls.' ripple-target" href="'.h($l['href']).'">';
    echo '          <span class="nav-ico"><i class="bi '.h($l['icon']).'"></i></span>';
    echo '          <span class="nav-text">'.h($l['label']).'</span>';
    echo '          <span class="nav-glow"></span>';
    echo '        </a>';
  }
  if (!empty($_SESSION['user_role']) && $_SESSION['user_role']==='admin') {
    echo '        <a class="'.($isActive('ia_settings.php')?'nav-item-link active':'nav-item-link').' ripple-target" href="'.$base.'/ia_settings.php">';
    echo '          <span class="nav-ico"><i class="bi bi-gear-wide-connected"></i></span>';
    echo '          <span class="nav-text">Config IA (APIs)</span>';
    echo '          <span class="nav-glow"></span>';
    echo '        </a>';
  }

  echo '      </div>';
  echo '    </div>';

  // ===== Configurações =====
  if (!empty($_SESSION['user_role']) && $_SESSION['user_role']==='admin' && perm_can('settings','view')) {
    echo '    <div class="nav-group" data-group="config">';
    echo '      <button class="nav-group-toggle ripple-target" type="button" data-group-btn="config">';
    echo '        <span class="nav-ico"><i class="bi bi-gear"></i></span>';
    echo '        <span class="nav-text">Configurações</span>';
    echo '        <span class="nav-right"><i class="bi bi-chevron-down nav-chevron"></i></span>';
    echo '      </button>';
    echo '      <div class="nav-group-items">';
    echo '        <a class="'.($isActive('settings.php')?'nav-item-link active':'nav-item-link').' ripple-target" href="'.$base.'/settings.php">';
    echo '          <span class="nav-ico"><i class="bi bi-palette2"></i></span>';
    echo '          <span class="nav-text">Branding/SMTP</span>';
    echo '          <span class="nav-glow"></span>';
    echo '        </a>';
    echo '      </div>';
    echo '    </div>';
  }

  echo '  </nav>';

  echo '  <div class="sidebar-footer">';
  echo '    <div class="user-chip">';
  echo '      <div class="user-dot"></div>';
  echo '      <div class="user-meta">';
  echo '        <div class="user-name">'.h($_SESSION['user_name'] ?? 'Usuário').'</div>';
  echo '        <div class="user-role">'.h($_SESSION['user_role'] ?? '').'</div>';
  echo '      </div>';
  echo '    </div>';
  echo '    <a class="btn btn-sm btn-danger w-100 mt-2 ripple-target" href="'.$base.'/logout.php"><i class="bi bi-box-arrow-right me-1"></i>Sair</a>';
  echo '  </div>';

  echo '</aside>';

  echo '<div class="app-main">';
  echo '  <header class="app-topbar">';
  echo '    <div class="d-flex align-items-center gap-2">';
  echo '      <button class="btn btn-sm btn-ghost icon-btn ripple-target d-lg-none" id="btnSidebar" type="button" title="Menu">';
  echo '        <i class="bi bi-list"></i>';
  echo '      </button>';
  echo '      <div class="topbar-title">'.h($title).'</div>';
  echo '    </div>';
  echo '    <div class="topbar-actions d-flex align-items-center gap-2">';
  echo '      <a class="btn btn-sm btn-ghost icon-btn ripple-target" href="'.$base.'/profile.php" title="Perfil"><i class="bi bi-person-circle"></i></a>';
  echo '      <button class="btn btn-sm btn-ghost icon-btn ripple-target" id="themeToggle" type="button" title="Tema">';
  echo '        <i class="bi bi-moon-stars" id="themeIcon"></i>';
  echo '      </button>';
  echo '    </div>';
  echo '  </header>';

  echo '  <main class="app-content">';
}

function render_footer(): void {
  $base = base_url_runtime();
  echo '  </main>';
  echo '</div>';
  echo '<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>';
  echo '<script src="'.$base.'/assets/app.js"></script>';
  echo '</body></html>';
}
