<?php
// app/finance.php

function split_installments(string $total, int $n): array {
  $totalCents = (int) round(((float)$total) * 100);
  $base = intdiv($totalCents, $n);
  $rem = $totalCents % $n;

  $arr = [];
  for ($i=1; $i<=$n; $i++) {
    $c = $base + ($i <= $rem ? 1 : 0);
    $arr[] = number_format($c/100, 2, '.', '');
  }
  return $arr;
}

function recalc_installment_status(PDO $pdo, int $installmentId): void {
  $st = $pdo->prepare("SELECT amount, paid_amount, status, document_id FROM fin_installments WHERE id=? FOR UPDATE");
  $st->execute([$installmentId]);
  $i = $st->fetch();
  if (!$i) throw new Exception('Parcela inválida.');
  if ($i['status'] === 'CANCELADO') return;

  $amount = (float)$i['amount'];
  $paid = (float)$i['paid_amount'];

  if ($paid <= 0.00001) $new = 'ABERTO';
  elseif ($paid + 0.00001 < $amount) $new = 'PARCIAL';
  else $new = 'BAIXADO';

  $pdo->prepare("UPDATE fin_installments SET status=?, updated_at=NOW(), settled_at=IF(?='BAIXADO', COALESCE(settled_at,NOW()), NULL) WHERE id=?")
      ->execute([$new, $new, $installmentId]);

  $chk = $pdo->prepare("SELECT SUM(status IN ('ABERTO','PARCIAL')) c FROM fin_installments WHERE document_id=?");
  $chk->execute([(int)$i['document_id']]);
  $open = (int)$chk->fetchColumn();

  $pdo->prepare("UPDATE fin_documents SET status=?, updated_at=NOW() WHERE id=?")
      ->execute([$open === 0 ? 'QUITADO' : 'ABERTO', (int)$i['document_id']]);
}

function settlement_create(int $installmentId, int $bankId, string $amount, string $note=''): int {
  $pdo = db();
  $pdo->beginTransaction();
  try {
    $st = $pdo->prepare("
      SELECT i.id, i.amount, i.paid_amount, i.status, d.doc_type, d.counterparty_name, d.description
      FROM fin_installments i
      JOIN fin_documents d ON d.id=i.document_id
      WHERE i.id=? FOR UPDATE
    ");
    $st->execute([$installmentId]);
    $row = $st->fetch();
    if (!$row) throw new Exception('Parcela não encontrada.');
    if ($row['status'] === 'CANCELADO') throw new Exception('Parcela cancelada.');
    if ($bankId <= 0) throw new Exception('Banco obrigatório.');

    $val = (float)number_format((float)$amount, 2, '.', '');
    if ($val <= 0) throw new Exception('Valor inválido.');

    $remaining = (float)$row['amount'] - (float)$row['paid_amount'];
    if ($val - 0.00001 > $remaining) throw new Exception('Valor maior que o restante da parcela.');

    $isReceber = ($row['doc_type'] === 'RECEBER');
    $direction = $isReceber ? 'CREDITO' : 'DEBITO';
    $delta = $isReceber ? $val : -$val;

    $pdo->prepare("
      INSERT INTO fin_settlements (installment_id, bank_id, direction, amount, note, occurred_at)
      VALUES (?,?,?,?,?,NOW())
    ")->execute([$installmentId, $bankId, $direction, $val, $note ?: null]);
    $settlementId = (int)$pdo->lastInsertId();

    $desc = ($isReceber ? 'Recebimento' : 'Pagamento') . " - {$row['counterparty_name']} - {$row['description']}";
    if ($note) $desc .= " | {$note}";
    $pdo->prepare("
      INSERT INTO bank_ledger (bank_id, settlement_id, entry_type, amount, description, occurred_at)
      VALUES (?,?,?,?,?,NOW())
    ")->execute([$bankId, $settlementId, $direction, $val, $desc]);

    $pdo->prepare("UPDATE banks SET current_balance=current_balance + :d, updated_at=NOW() WHERE id=:id")
        ->execute([':d'=>$delta, ':id'=>$bankId]);

    $pdo->prepare("UPDATE fin_installments SET paid_amount = paid_amount + :v, bank_id=:bank, updated_at=NOW() WHERE id=:id")
        ->execute([':v'=>$val, ':bank'=>$bankId, ':id'=>$installmentId]);

    recalc_installment_status($pdo, $installmentId);

    audit_log('create','settlement',$settlementId,['installment_id'=>$installmentId,'bank_id'=>$bankId,'amount'=>$val]);
    $pdo->commit();
    return $settlementId;
  } catch(Throwable $e) {
    $pdo->rollBack();
    throw $e;
  }
}

function settlement_reverse(int $settlementId, string $note=''): void {
  $pdo = db();
  $pdo->beginTransaction();
  try {
    $st = $pdo->prepare("
      SELECT s.*, i.id AS inst_id, d.doc_type, d.counterparty_name, d.description
      FROM fin_settlements s
      JOIN fin_installments i ON i.id=s.installment_id
      JOIN fin_documents d ON d.id=i.document_id
      WHERE s.id=? FOR UPDATE
    ");
    $st->execute([$settlementId]);
    $s = $st->fetch();
    if (!$s) throw new Exception('Baixa não encontrada.');
    if ($s['reversed_at'] !== null) throw new Exception('Já estornada.');

    $val = (float)$s['amount'];
    $isReceber = ($s['direction'] === 'CREDITO');
    $invType = $isReceber ? 'DEBITO' : 'CREDITO';
    $delta = $isReceber ? -$val : +$val;

    $desc = "Estorno - " . ($isReceber ? "Recebimento" : "Pagamento") . " - {$s['counterparty_name']} - {$s['description']}";
    if ($note) $desc .= " | {$note}";

    $pdo->prepare("
      INSERT INTO bank_ledger (bank_id, settlement_id, entry_type, amount, description, occurred_at)
      VALUES (?,?,?,?,?,NOW())
    ")->execute([(int)$s['bank_id'], $settlementId, $invType, $val, $desc]);

    $pdo->prepare("UPDATE banks SET current_balance=current_balance + :d, updated_at=NOW() WHERE id=:id")
        ->execute([':d'=>$delta, ':id'=>(int)$s['bank_id']]);

    $pdo->prepare("UPDATE fin_settlements SET reversed_at=NOW() WHERE id=?")->execute([$settlementId]);

    $pdo->prepare("UPDATE fin_installments SET paid_amount = GREATEST(paid_amount - :v, 0), updated_at=NOW() WHERE id=:id")
        ->execute([':v'=>$val, ':id'=>(int)$s['inst_id']]);

    recalc_installment_status($pdo, (int)$s['inst_id']);
    audit_log('reverse','settlement',$settlementId,['bank_id'=>(int)$s['bank_id'],'amount'=>$val]);
    $pdo->commit();
  } catch(Throwable $e) {
    $pdo->rollBack();
    throw $e;
  }
}
