<?php
// app/helpers.php
function h(string $s): string { return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }

/**
 * Base URL resiliente.
 * - Se app.base_url estiver definido, usa ele
 * - Caso contrário, detecta automaticamente a partir de /public (dirname de SCRIPT_NAME)
 */
function perm_can(string $key, string $action='view'): bool {
  $role = $_SESSION['user_role'] ?? '';
  if ($role === 'admin') return true;
  $perms = $_SESSION['perms'] ?? [];
  $p = $perms[$key] ?? null;
  if (!$p) return false;
  return (bool)($p[$action] ?? false);
}

function require_perm(string $key, string $action='view'): void {
  if (!perm_can($key, $action)) {
    http_response_code(403);
    die('Sem permissão.');
  }
}

function base_url_runtime(): string {
  $cfg = require __DIR__ . '/config.php';
  $forced = rtrim((string)($cfg['app']['base_url'] ?? ''), '/');
  if ($forced !== '') return $forced;
  $script = (string)($_SERVER['SCRIPT_NAME'] ?? '');
  $dir = rtrim(str_replace('\\', '/', dirname($script)), '/');
  if ($dir === '.' || $dir === '/') return '';
  return $dir;
}


function redirect(string $path) {
  $base = base_url_runtime();
  header("Location: {$base}{$path}");
  exit;
}

function csrf_token(): string {
  if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(32));
  return $_SESSION['csrf'];
}

function csrf_check(): void {
  $token = $_POST['csrf'] ?? '';
  if (!$token || empty($_SESSION['csrf']) || !hash_equals($_SESSION['csrf'], $token)) {
    http_response_code(403);
    die('CSRF inválido.');
  }
}

function money_to_decimal(string $s): string {
  $s = trim($s);
  $s = str_replace(['R$', ' '], '', $s);
  if (str_contains($s, ',')) {
    $s = str_replace('.', '', $s);
    $s = str_replace(',', '.', $s);
  }
  if (!is_numeric($s)) return '0.00';
  return number_format((float)$s, 2, '.', '');
}

function render_header(string $title): void {
  $base = base_url_runtime();
  $appName = function_exists('setting_get') ? (string)(setting_get('app.name', 'Financeiro') ?: 'Financeiro') : 'Financeiro';
  $logoPath = function_exists('setting_get') ? (string)(setting_get('app.logo_path', '') ?: '') : '';

  $page = basename($_SERVER['PHP_SELF'] ?? '');
  $qs = $_SERVER['QUERY_STRING'] ?? '';
  $full = $page . ($qs ? ('?'.$qs) : '');

  $isActive = function(string $needle) use ($page, $full): bool {
    return str_contains($page, $needle) || str_contains($full, $needle);
  };

  // Badges dinâmicos (silencioso em caso de erro)
  $badgePagar = 0;
  $badgeReceber = 0;
  $badgeRecon = 0;
  try {
    if (function_exists('db')) {
      $pdo = db();
      $st = $pdo->query("
        SELECT
          SUM(d.doc_type='PAGAR' AND i.status IN ('ABERTO','PARCIAL')) AS pagar_open,
          SUM(d.doc_type='RECEBER' AND i.status IN ('ABERTO','PARCIAL')) AS receber_open
        FROM fin_installments i
        JOIN fin_documents d ON d.id=i.document_id
      ");
      $r = $st->fetch();
      if ($r) {
        $badgePagar = (int)($r['pagar_open'] ?? 0);
        $badgeReceber = (int)($r['receber_open'] ?? 0);
      }

      $st2 = $pdo->query("
        SELECT COUNT(*) AS c
        FROM bank_statement_entries se
        LEFT JOIN bank_reconciliations br ON br.statement_entry_id = se.id
        WHERE br.id IS NULL
      ");
      $badgeRecon = (int)($st2->fetch()['c'] ?? 0);
    }
  } catch (Throwable $e) {}

  $badgeHtml = function(int $n, string $variant='info'): string {
    if ($n <= 0) return '';
    $v = h($variant);
    return '<span class="nav-badge nav-badge-'.$v.'">'.(int)$n.'</span>';
  };

  echo '<!doctype html><html lang="pt-br" data-theme="dark"><head>';
  echo '<meta charset="utf-8"><meta name="viewport" content="width=device-width, initial-scale=1">';
  echo '<title>'.h($title).'</title>';
  echo '<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">';
  echo '<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">';
  echo '<link href="'.$base.'/assets/app.css" rel="stylesheet">';
  echo '</head><body class="app-shell no-js">';

  echo '<div class="app-overlay" id="appOverlay"></div>';

  echo '<aside class="app-sidebar" id="appSidebar">';
  echo '  <div class="sidebar-brand">';
  echo '    <a class="brand-link ripple-target" href="'.$base.'/index.php">';
  if ($logoPath) echo '      <img class="brand-logo" src="'.h($base.$logoPath).'" alt="logo">';
  echo '      <span class="brand-name">'.h($appName).'</span>';
  echo '    </a>';
  echo '    <div class="brand-actions">';
  echo '      <button class="btn btn-sm btn-ghost icon-btn ripple-target d-none d-lg-inline" id="btnCollapse" type="button" title="Recolher">';
  echo '        <i class="bi bi-layout-sidebar-inset"></i>';
  echo '      </button>';
  echo '    </div>';
  echo '  </div>';

  echo '  <div class="sidebar-search px-3 pt-2 pb-2">';
  echo '    <div class="input-group input-group-sm">';
  echo '      <span class="input-group-text bg-transparent text-white-50 border-0"><i class="bi bi-search"></i></span>';
  echo '      <input class="form-control form-control-sm sidebar-search-input" placeholder="Buscar..." id="sidebarSearch">';
  echo '    </div>';
  echo '  </div>';

  echo '  <nav class="sidebar-nav" id="sidebarNav">';

  // FINANCEIRO
  echo '    <div class="nav-group open" data-group="financeiro">';
  echo '      <button class="nav-group-toggle ripple-target" type="button" data-group-btn="financeiro">';
  echo '        <span class="nav-ico"><i class="bi bi-cash-stack"></i></span>';
  echo '        <span class="nav-text">Financeiro</span>';
  echo '        <span class="nav-right">';
  echo '          <span class="nav-badges">'.$badgeHtml($badgePagar + $badgeReceber, 'primary').'</span>';
  echo '          <i class="bi bi-chevron-down nav-chevron"></i>';
  echo '        </span>';
  echo '      </button>';
  echo '      <div class="nav-group-items">';
    
    
    
    

    echo '          <span class="nav-glow"></span>';
    echo '        </a>';

  echo '        <a class="'.(str_contains($full,'contas.php?type=PAGAR')?'nav-item-link active':'nav-item-link').' ripple-target" href="'.$base.'/contas.php?type=PAGAR">';
  echo '          <span class="nav-ico"><i class="bi bi-arrow-down-circle"></i></span>';
  echo '          <span class="nav-text">Contas a Pagar</span>';
  echo            $badgeHtml($badgePagar, 'danger');
  echo '          <span class="nav-glow"></span>';
  echo '        </a>';
  echo '        <a class="'.(str_contains($full,'contas.php?type=RECEBER')?'nav-item-link active':'nav-item-link').' ripple-target" href="'.$base.'/contas.php?type=RECEBER">';
  echo '          <span class="nav-ico"><i class="bi bi-arrow-up-circle"></i></span>';
  echo '          <span class="nav-text">Contas a Receber</span>';
  echo            $badgeHtml($badgeReceber, 'success');
  echo '          <span class="nav-glow"></span>';
  echo '        </a>';
  echo '      </div>';
  echo '    </div>';

  // CADASTROS
  echo '    <div class="nav-group" data-group="cadastros">';
  echo '      <button class="nav-group-toggle ripple-target" type="button" data-group-btn="cadastros">';
  echo '        <span class="nav-ico"><i class="bi bi-ui-checks-grid"></i></span>';
  echo '        <span class="nav-text">Cadastros</span>';
  echo '        <span class="nav-right"><i class="bi bi-chevron-down nav-chevron"></i></span>';
  echo '      </button>';
  echo '      <div class="nav-group-items">';

  $cad = [
    ['href'=>$base.'/banks.php', 'label'=>'Bancos', 'icon'=>'bi-bank', 'active'=>$isActive('banks.php')],
    ['href'=>$base.'/parties.php', 'label'=>'Clientes/Fornecedores', 'icon'=>'bi-people', 'active'=>$isActive('parties.php')],
    ['href'=>$base.'/parties_import.php', 'label'=>'Importar CSV', 'icon'=>'bi-upload', 'active'=>$isActive('parties_import.php')],
    ['href'=>$base.'/categories.php', 'label'=>'Categorias', 'icon'=>'bi-tags', 'active'=>$isActive('categories.php')],
    ['href'=>$base.'/cost_centers.php', 'label'=>'Centros de Custo', 'icon'=>'bi-diagram-3', 'active'=>$isActive('cost_centers.php')],
    ['href'=>$base.'/accounts.php', 'label'=>'Plano de Contas', 'icon'=>'bi-list-check', 'active'=>$isActive('accounts.php')],
  ];
  foreach ($cad as $l) {
    $cls = $l['active'] ? 'nav-item-link active' : 'nav-item-link';
    echo '        <a class="'.$cls.' ripple-target" href="'.h($l['href']).'">';
    echo '          <span class="nav-ico"><i class="bi '.h($l['icon']).'"></i></span>';
    echo '          <span class="nav-text">'.h($l['label']).'</span>';
    echo '          <span class="nav-glow"></span>';
    echo '        </a>';
  }
  echo '      </div>';
  echo '    </div>';

  // RELATÓRIOS
  echo '    <div class="nav-group" data-group="relatorios">';
  echo '      <button class="nav-group-toggle ripple-target" type="button" data-group-btn="relatorios">';
  echo '        <span class="nav-ico"><i class="bi bi-bar-chart-line"></i></span>';
  echo '        <span class="nav-text">Relatórios</span>';
  echo '        <span class="nav-right">'.$badgeHtml($badgeRecon, 'warning').'<i class="bi bi-chevron-down nav-chevron"></i></span>';
  echo '      </button>';
  echo '      <div class="nav-group-items">';

  $rep = [
    ['href'=>$base.'/index.php', 'label'=>'Dashboard', 'icon'=>'bi-speedometer2', 'active'=>$isActive('index.php')],
    ['href'=>$base.'/dre.php', 'label'=>'DRE', 'icon'=>'bi-graph-up', 'active'=>$isActive('dre.php')],
    ['href'=>$base.'/reconcile_import.php', 'label'=>'Conciliação', 'icon'=>'bi-check2-circle', 'active'=>$isActive('reconcile_'), 'badge'=>$badgeRecon],
    ];
  foreach ($rep as $l) {
    $cls = $l['active'] ? 'nav-item-link active' : 'nav-item-link';
    echo '        <a class="'.$cls.' ripple-target" href="'.h($l['href']).'">';
    echo '          <span class="nav-ico"><i class="bi '.h($l['icon']).'"></i></span>';
    echo '          <span class="nav-text">'.h($l['label']).'</span>';
    if (!empty($l['badge'])) echo $badgeHtml((int)$l['badge'], 'warning');
    echo '          <span class="nav-glow"></span>';
    echo '        </a>';
  }

  echo '      </div>';
  echo '    </div>';

  
  // IA
  echo '    <div class="nav-group" data-group="ia">';
  echo '      <button class="nav-group-toggle ripple-target" type="button" data-group-btn="ia">';
  echo '        <span class="nav-ico"><i class="bi bi-stars"></i></span>';
  echo '        <span class="nav-text">IA</span>';
  echo '        <span class="nav-right"><i class="bi bi-chevron-down nav-chevron"></i></span>';
  echo '      </button>';
  echo '      <div class="nav-group-items">';
  echo '        <a class="'.($isActive('assistant_ai.php')?'nav-item-link active':'nav-item-link').' ripple-target" href="'.$base.'/assistant_ai.php">';
  echo '          <span class="nav-ico"><i class="bi bi-robot"></i></span>';
  echo '          <span class="nav-text">Assistente IA</span>';
  echo '          <span class="nav-glow"></span>';
  echo '        </a>';
  echo '        <a class="'.($isActive('prompt_categories.php')?'nav-item-link active':'nav-item-link').' ripple-target" href="'.$base.'/prompt_categories.php">';
  echo '          <span class="nav-ico"><i class="bi bi-folder2"></i></span>';
  echo '          <span class="nav-text">Categorias de Prompt</span>';
  echo '          <span class="nav-glow"></span>';
  echo '        </a>';
  echo '        <a class="'.($isActive('prompts.php')?'nav-item-link active':'nav-item-link').' ripple-target" href="'.$base.'/prompts.php">';
  echo '          <span class="nav-ico"><i class="bi bi-card-text"></i></span>';
  echo '          <span class="nav-text">Prompts</span>';
  echo '          <span class="nav-glow"></span>';
  echo '        </a>';
  if (!empty($_SESSION['user_role']) && $_SESSION['user_role']==='admin') {
    echo '        <a class="'.($isActive('ia_settings.php')?'nav-item-link active':'nav-item-link').' ripple-target" href="'.$base.'/ia_settings.php">';
    echo '          <span class="nav-ico"><i class="bi bi-gear-wide-connected"></i></span>';
    echo '          <span class="nav-text">Config IA (APIs)</span>';
    echo '          <span class="nav-glow"></span>';
    echo '        </a>';
  }
  echo '      </div>';
  echo '    </div>';


// ADMIN
  if (!empty($_SESSION['user_role']) && $_SESSION['user_role'] === 'admin') {
    echo '    <div class="nav-group" data-group="admin">';
    echo '      <button class="nav-group-toggle ripple-target" type="button" data-group-btn="admin">';
    echo '        <span class="nav-ico"><i class="bi bi-shield-lock"></i></span>';
    echo '        <span class="nav-text">Admin</span>';
    echo '        <span class="nav-right"><i class="bi bi-chevron-down nav-chevron"></i></span>';
    echo '      </button>';
    echo '      <div class="nav-group-items">';
    $cls = $isActive('settings.php') ? 'nav-item-link active' : 'nav-item-link';
    echo '        <a class="'.$cls.' ripple-target" href="'.$base.'/settings.php">';
    echo '          <span class="nav-ico"><i class="bi bi-gear"></i></span>';
    echo '          <span class="nav-text">Configurações</span>';
    echo '          <span class="nav-glow"></span>';
    echo '        </a>';
    echo '      </div>';
    echo '    </div>';
  }

  echo '  </nav>';

  echo '  <div class="sidebar-footer">';
  echo '    <div class="user-chip">';
  echo '      <div class="user-dot"></div>';
  echo '      <div class="user-meta">';
  echo '        <div class="user-name">'.h($_SESSION['user_name'] ?? 'Usuário').'</div>';
  echo '        <div class="user-role">'.h($_SESSION['user_role'] ?? '').'</div>';
  echo '      </div>';
  echo '    </div>';
  echo '    <a class="btn btn-sm btn-danger w-100 mt-2 ripple-target" href="'.$base.'/logout.php"><i class="bi bi-box-arrow-right me-1"></i>Sair</a>';
  echo '  </div>';

  echo '</aside>';

  echo '<div class="app-main">';
  echo '  <header class="app-topbar">';
  echo '    <div class="d-flex align-items-center gap-2">';
  echo '      <button class="btn btn-sm btn-ghost icon-btn ripple-target d-lg-none" id="btnSidebar" type="button" title="Menu">';
  echo '        <i class="bi bi-list"></i>';
  echo '      </button>';
  echo '      <div class="topbar-title">'.h($title).'</div>';
  echo '    </div>';
  echo '    <div class="topbar-actions d-flex align-items-center gap-2">';
  echo '      <a class="btn btn-sm btn-ghost icon-btn ripple-target" href="'.$base.'/profile.php" title="Perfil"><i class="bi bi-person-circle"></i></a>';
  echo '      <button class="btn btn-sm btn-ghost icon-btn ripple-target" id="themeToggle" type="button" title="Tema">';
  echo '        <i class="bi bi-moon-stars" id="themeIcon"></i>';
  echo '      </button>';
  echo '      <a class="btn btn-sm btn-outline-light ripple-target" href="'.$base.'/conta_form.php?type=PAGAR"><i class="bi bi-plus-circle me-1"></i>Pagar</a>';
  echo '      <a class="btn btn-sm btn-outline-light ripple-target" href="'.$base.'/conta_form.php?type=RECEBER"><i class="bi bi-plus-circle me-1"></i>Receber</a>';
  echo '    </div>';
  echo '  </header>';

  echo '  <main class="app-content">';
}

function render_footer(): void {
  $base = base_url_runtime();
  echo '  </main>';
  echo '</div>';

  echo '<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>';
  echo '<script src="'.$base.'/assets/app.js"></script>';
  echo '</body></html>';
}
